package nemosofts.voxradio.adapter;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.view.ContextThemeWrapper;
import androidx.appcompat.widget.PopupMenu;
import androidx.media3.common.PlaybackException;
import androidx.media3.common.Player;
import androidx.nemosofts.theme.ThemeEngine;
import androidx.recyclerview.widget.RecyclerView;

import java.io.File;
import java.text.DecimalFormat;
import java.util.List;

import nemosofts.voxradio.R;
import nemosofts.voxradio.callback.Callback;
import nemosofts.voxradio.dialog.DialogUtil;
import nemosofts.voxradio.item.ItemRecorder;
import nemosofts.voxradio.utils.RecorderPlayer;

public class AdapterRecorder extends RecyclerView.Adapter<AdapterRecorder.MyViewHolder> {

    private final Context context;
    private final List<ItemRecorder> arrayList;
    private final RecyclerItemClickListener listener;
    private final RecorderPlayer recorderPlayer;

    public static class MyViewHolder extends RecyclerView.ViewHolder {

        private final TextView title;
        private final TextView artist;
        private final TextView views;
        private final ImageView play;
        private final ImageView pause;
        private final ImageView option;

        MyViewHolder(View view) {
            super(view);
            title = view.findViewById(R.id.tv_songlist_name);
            artist =  view.findViewById(R.id.tv_songlist_cat);
            play = view.findViewById(R.id.play);
            pause = view.findViewById(R.id.pause);
            views = view.findViewById(R.id.tv_songlist_vie);
            option = view.findViewById(R.id.iv_option);
        }
    }

    public AdapterRecorder(Context context,
                           List<ItemRecorder> arrayList,
                           RecyclerItemClickListener listener) {
        this.arrayList = arrayList;
        this.context = context;
        this.listener = listener;
        recorderPlayer = new RecorderPlayer(context);
        recorderPlayer.onCreate();
    }

    @NonNull
    @Override
    public MyViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View itemView = LayoutInflater.from(parent.getContext())
                .inflate(R.layout.row_recorder, parent, false);
        return new MyViewHolder(itemView);
    }

    @Override
    public void onBindViewHolder(@NonNull final MyViewHolder holder, @SuppressLint("RecyclerView") int position) {
        final ItemRecorder item = arrayList.get(position);

        holder.title.setText(getTitleData(item.getTitle()));
        holder.artist.setText(item.getDuration());
        holder.views.setText(getStringSizeLengthFile(item.getFilesets()));

        final String finalUrl =  item.getMp3();

        RecorderPlayer.getExoPlayer().addListener(new Player.Listener() {

            @Override
            public void onPlaybackStateChanged(int playbackState) {
                Player.Listener.super.onPlaybackStateChanged(playbackState);
                if (playbackState == Player.STATE_ENDED) {
                    holder.play.setVisibility(View.VISIBLE);
                    holder.pause.setVisibility(View.GONE);
                }
            }

            @Override
            public void onPlayerError(@NonNull PlaybackException error) {
                Player.Listener.super.onPlayerError(error);
                RecorderPlayer.getExoPlayer().setPlayWhenReady(false);
            }
        });

        holder.pause.setOnClickListener(view -> {
            if (RecorderPlayer.getExoPlayer().getPlayWhenReady()) {
                RecorderPlayer.getExoPlayer().setPlayWhenReady(false);
                holder.pause.setImageResource(R.drawable.ic_play);
            } else {
                RecorderPlayer.getExoPlayer().setPlayWhenReady(true);
                holder.pause.setImageResource(R.drawable.ic_pause);
            }
        });

        holder.play.setOnClickListener(view -> {
            listener.onClickListener(item, position);
            recorderPlayer.setUrl(finalUrl);
            holder.pause.setImageResource(R.drawable.ic_pause);
            holder.play.setImageResource(R.drawable.ic_play);
            holder.play.setVisibility(View.GONE);
            holder.pause.setVisibility(View.VISIBLE);

        });

        if (RecorderPlayer.getIsPlaying() && Callback.getPlayPosRecording() <= holder.getAbsoluteAdapterPosition()
                && Callback.getArrayListRecording().get(Callback.getPlayPosRecording()).getId().equals(arrayList.get(position).getId())) {
            holder.play.setVisibility(View.GONE);
            holder.pause.setVisibility(View.VISIBLE);
        } else {
            holder.pause.setVisibility(View.GONE);
            holder.play.setVisibility(View.VISIBLE);
        }

        holder.option.setOnClickListener(view -> openOptionPopUp(holder.option, holder.getAbsoluteAdapterPosition(), item, listener));
    }

    @NonNull
    public static String getStringSizeLengthFile(long size) {
        DecimalFormat df = new DecimalFormat("0.00");
        float sizeKb = 1024.0f;
        float sizeMb = sizeKb * sizeKb;
        float sizeGb = sizeMb * sizeKb;
        float sizeTerra = sizeGb * sizeKb;
        if(size < sizeMb)
            return df.format(size / sizeKb)+ " Kb";
        else if(size < sizeGb)
            return df.format(size / sizeMb) + " Mb";
        else if(size < sizeTerra)
            return df.format(size / sizeGb) + " Gb";
        return "";
    }

    private void openOptionPopUp(ImageView option, final int pos, ItemRecorder itemRecorder,
                                 RecyclerItemClickListener listener) {
        ContextThemeWrapper ctw = new ContextThemeWrapper(context, new ThemeEngine(context).getIsThemeMode()
                ? R.style.PopupMenuDark
                : R.style.PopupMenuLight
        );
        PopupMenu popup = new PopupMenu(ctw, option);
        popup.getMenuInflater().inflate(R.menu.popup_song_off, popup.getMenu());
        popup.setForceShowIcon(true);
        popup.setOnMenuItemClickListener(item -> {
            if (item.getItemId() == R.id.popup_delete) {
                openDeleteDialog(pos, listener);
            }else if (item.getItemId() == R.id.popup_share) {
                try {
                    Intent share = new Intent(Intent.ACTION_SEND);
                    share.setType("audio/mp3");
                    share.putExtra(Intent.EXTRA_STREAM, Uri.parse(itemRecorder.getMp3()));
                    share.putExtra(Intent.EXTRA_TEXT, context.getResources().getString(R.string.listening) + " - "
                            + getTitleData(itemRecorder.getTitle()) + "\n\nvia "
                            + context.getResources().getString(R.string.app_name)
                            + " - http://play.google.com/store/apps/details?id=" + context.getPackageName());
                    context.startActivity(Intent.createChooser(share, context.getResources().getString(R.string.share_record)));
                } catch (Exception e) {
                    Log.e("AdapterRecorder","Error share" ,e);
                }
            }
            return true;
        });
        popup.show();
    }

    @NonNull
    public static String getTitleData(@NonNull String title) {
        String titlePath;
        titlePath = title.replace(".mp3", "");
        return titlePath;
    }

    @SuppressLint("NotifyDataSetChanged")
    private void openDeleteDialog(final int pos, RecyclerItemClickListener listener) {
        DialogUtil.trashDialog((Activity) context, new DialogUtil.DeleteListener() {
            @Override
            public void onDelete() {
                deleteFiles(arrayList.get(pos).getMp3());
                arrayList.remove(pos);
                notifyItemRemoved(pos);
                listener.onDeleteListener();
            }

            @Override
            public void onCancel() {
                // this method is empty
            }
        });
    }

    private void deleteFiles(String path) {
        try {
            File file = new File(path);
            boolean isDeleted = file.exists() && file.delete();
            if (isDeleted) {
                Toast.makeText(context, context.getString(R.string.file_deleted), Toast.LENGTH_SHORT).show();
            }
        } catch (Exception e) {
            Log.e("FileDeletion", "Error deleting file: " + path, e);
        }
    }

    @Override
    public long getItemId(int id) {
        return id;
    }

    @Override
    public int getItemCount() {
        return arrayList.size();
    }

    public interface RecyclerItemClickListener{
        void onClickListener(ItemRecorder itemData, int position);
        void onDeleteListener();
    }
}